// RadarWnd.c
// This file contains the window procedure and code related to the
// radar window. The radar indicates the relative position of the
// tank or robot.

#include <windows.h>	// Normal Windows stuff
#include <math.h>		// sine and cosine functions
#include <gl\gl.h>
#include <gl\glu.h>
#include "glutils.h"	
#include "externs.h"	// Data shared between files

// Local functions
void DrawBlip(HDC hDC, int nCenterX, int nCenterY);

static int nRadarRadius[4] = { 70, 50, 30, 10 };


////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
// Window procedure, handles all messages for this window
LRESULT CALLBACK WndProcRadar(HWND    hWnd,
							UINT    message,
							WPARAM  wParam,
							LPARAM  lParam)
	{
	switch (message)
		{
		// Variables that need persistent storage
		static double dSweepAngle = 0.0f;
		static HPEN hPen = NULL;
		static HBRUSH hBrush = NULL;

		// Window creation, setup here
		case WM_CREATE:
			// Create a bright green drawing pen, and a darker green
			// for the background of the radar
			hPen = CreatePen(PS_SOLID,2,RGB(0,255,0));
			hBrush = CreateSolidBrush(RGB(0,128,0));

			// Set a timer, this timer moves the radar arm around in circles.
			SetTimer(hWnd,101,125,NULL);

			break;

		// Window is being destroyed, cleanup
		case WM_DESTROY:
			// First get rid of the timer
			KillTimer(hWnd, 101);

			// Delete the window objects we created (pen and brush)
			DeleteObject(hPen);
			DeleteObject(hBrush);
			break;


		// Draw the radar screen, if the tank/robot is in proximity, 
		// put a blip in the appropriate position.
		case WM_PAINT:
			{
			PAINTSTRUCT ps;
			RECT	rect;			// Dimensions of the window
			int nCenterX,nCenterY;	// Coordinates of center of radar	
			int i;					// Loop index
			HPEN hOldPen;			// Holder for deselected pen
			HBRUSH hOldBrush;		// Holder for deselected brush

			// Get the dimensions of the window and calculate center
			GetClientRect(hWnd, &rect);
			nCenterX = rect.right/2;
			nCenterY = rect.bottom/2;

			// Start drawing
			BeginPaint(hWnd,&ps);
			
			// Use our new pen and brush
			hOldPen = SelectObject(ps.hdc,hPen);
			hOldBrush = SelectObject(ps.hdc,hBrush);

			// Rings of Radar Screen
			for(i = 0; i < 4; i++)
				Ellipse(ps.hdc,
					nCenterX-nRadarRadius[i],	// nLeftRect
					nCenterY+nRadarRadius[i],	// nTopRect
					nCenterX+nRadarRadius[i],	// nRightRect
					nCenterY-nRadarRadius[i]);	// nBottomRect

			// Draw Sweeper
			MoveToEx(ps.hdc,nCenterX,nCenterY,NULL);
			LineTo(ps.hdc,nCenterX+(int)((double)nRadarRadius[0]*sin(dSweepAngle)),
						  nCenterY+(int)((double)nRadarRadius[0]*cos(dSweepAngle)));


			DrawBlip(ps.hdc, nCenterX, nCenterY);

			// Deselect the pen and brush
			SelectObject(ps.hdc,hOldPen);
			SelectObject(ps.hdc,hOldBrush);
	
			// Done painting
			EndPaint(hWnd, &ps);

			// Advance Sweeper for next timer pulse
			dSweepAngle -= PI/90.0f;
			if(dSweepAngle < 0.0f)
				dSweepAngle = 2.0f*PI;
			}
			break;

		// Just invalidate. WM_PAINT does all the work
		case WM_TIMER:
			InvalidateRect(hWnd,NULL,FALSE);
			break;

		default:   // Passes it on if unproccessed
			return (DefWindowProc(hWnd, message, wParam, lParam));

		}

	return (0L);
	}


///////////////////////////////////////////////////////////////////////////
// Draw a blip on the radar screen if we are within 700 units of the other object
// The 700 comes from the radar outer radius and can be changed by changing it.
void DrawBlip(HDC hDC, int nCenterX, int nCenterY)
	{
	double dDistance;
	struct _POSITION *pOtherObject;
	double xDelta, zDelta;
	double dAngle;
		
	// Who are WE and who is the "other" object
	if(bTank)						
		pOtherObject = &robotPos;	// We are the tank
	else
		pOtherObject = &tankPos;	// We are the robot

	// Calculate change in x and z coordinates to get to the other object
	xDelta = pOtherObject->xPos - pObject->xPos;
	zDelta = pOtherObject->zPos - pObject->zPos;
	
	// Calculate the distance
	dDistance = sqrt((xDelta*xDelta) + (zDelta*zDelta))/10.0;

	// Is it in range?
	if(dDistance > (nRadarRadius[0]-6))	// Bring in a little to keep off edge
		return;

	// Adjust position by rotating the coordinates.
	dAngle = atan2(zDelta,xDelta);	// This is the angle from direct east

	// Offset by rotation
	dAngle += pObject->radsFromEast-(PI/2.0);
		
	nCenterX += (int)(cos(dAngle)*dDistance);
	nCenterY += (int)(sin(dAngle)*dDistance);

	// Setup text parameters and draw an asterisk '*'
	SetBkMode(hDC,TRANSPARENT);
	SetTextColor(hDC,RGB(255,255,255));
	// Try to adjust center of * to match desired coordinate
	TextOut(hDC,nCenterX-3, nCenterY-3, "*",1);
	}



