// ViewWnd.c
// This file contains the window procedure and code related to the
// view window. This window does the OpenGL rendering of the terrain.

#include <windows.h>	// Normal Windows stuff
#include <math.h>
#include <gl/gl.h>		// Core OpenGL functions
#include <gl/glu.h>		// OpenGL Utility functions
#include <gl/glaux.h>
#include "externs.h"	// Data shared between files
#include "glutils.h"	// Utilities for this application


///////////////////////////////////////////////////////////////////////////////
// Setup the main view windows Rendering Context
void SetupViewRC(void)
	{
	// Black background
	glClearColor(0.0f, 0.0f, 0.0f, 1.0f );

	// Only draw the outside of CW wound objects
	glCullFace(GL_BACK);	// Cull the back
	glFrontFace(GL_CCW);	// Counter Clock wise wound is front
	glEnable(GL_CULL_FACE); // Enable the culling

	// Do depth testing
	glEnable(GL_DEPTH_TEST);

	}



////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
// Window procedure, handles all messages for this window
LRESULT CALLBACK WndProcView(HWND    hWnd,
							UINT    message,
							WPARAM  wParam,
							LPARAM  lParam)
	{
	static HDC	  hDC;	// Keep the Device Context
	static HGLRC  hRC;	// Keep the Rendering Context
	
	switch (message)
		{
		// Window creation, setup here
		case WM_CREATE:
			// Save the device context
			hDC = GetDC(hWnd);

			// Set the pixel format
			SetDCDepthPixelFormat(hDC);

			// Create palette if needed, this will return NULL if 
			// no palette is required
			hPalette = GetOpenGLPalette(hDC);
	
			// Create the rendering context and make it current
			hRC = wglCreateContext(hDC);
			wglMakeCurrent(hDC, hRC);

			// Do some setup here
			SetupViewRC();

			wglMakeCurrent(hDC,NULL);

			break;

		// Window is being destroyed, cleanup
		case WM_DESTROY:
			// Cleanup...
			// Deselect the current rendering context and delete it
			wglMakeCurrent(hDC,NULL);
			wglDeleteContext(hRC);

			// Destroy the palette if it was created 
			if(hPalette != NULL)
				DeleteObject(hPalette);

			// Release the device context
			ReleaseDC(hWnd,hDC);
			break;

		// Window is resized. Setup the viewing transformation
		case WM_SIZE:
			{
			int nWidth,nHeight;
			double dAspect;

			nWidth = LOWORD(lParam);  // width of client area 
			nHeight = HIWORD(lParam); // height of client area 
	
			if(nHeight == 0)		  // Don't allow divide by zero
				nHeight = 1;

			dAspect = (double)nWidth/(double)nHeight;

			// Make this rendering context current
			wglMakeCurrent(hDC, hRC);

			// Set the viewport to be the entire window
		    glViewport(0, 0, nWidth, nHeight);
	
			// Setup Perspective
			glMatrixMode(GL_PROJECTION);
			glLoadIdentity();

			// Establish viewing volume
			gluPerspective(35.0, dAspect,1, 2000);

			glMatrixMode(GL_MODELVIEW);
			glLoadIdentity();

			wglMakeCurrent(hDC,NULL);
			}
			break;

		case WM_PAINT:
			{
			float fRadius = 50.0f;

			wglMakeCurrent(hDC, hRC);

			// Clear the window with current clearing color
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

			glPushMatrix();

			// Handy dandy little function
			gluLookAt(pObject->xPos, 5.0f, pObject->zPos,	// Eye Position (x,y,z)
				pObject->xPos + (fRadius*(float)cos(pObject->radsFromEast)), // Center X
				5.0f,															// Center Y
				pObject->zPos - (fRadius*(float)sin(pObject->radsFromEast)), // Center Z
	
				0.0f, 1.0f, 0.0f);						   // Up is straight up Y axis


			RenderWorld();

			glPopMatrix();

			SwapBuffers(hDC);

			wglMakeCurrent(hDC,NULL);

			// Validate the newly painted client area
			ValidateRect(hWnd,NULL);
			}
			break;


		// Windows is telling the application that it may modify
		// the system palette.  This message in essance asks the 
		// application for a new palette.
		case WM_QUERYNEWPALETTE:
			// If the palette was created.
			if(hPalette)
				{
				int nRet;

				// Selects the palette into the current device context
				SelectPalette(hDC, hPalette, FALSE);

				// Map entries from the currently selected palette to
				// the system palette.  The return value is the number 
				// of palette entries modified.
				nRet = RealizePalette(hDC);

				// Repaint, forces remap of palette in current window
				InvalidateRect(hWnd,NULL,FALSE);

				return nRet;
				}
			break;

	
		// This window may set the palette, even though it is not the 
		// currently active window.
		case WM_PALETTECHANGED:
			// Don't do anything if the palette does not exist, or if
			// this is the window that changed the palette.
			if((hPalette != NULL) && ((HWND)wParam != hWnd))
				{
				// Select the palette into the device context
				SelectPalette(hDC,hPalette,FALSE);

				// Map entries to system palette
				RealizePalette(hDC);
				
				// Remap the current colors to the newly realized palette
				UpdateColors(hDC);
				return 0;
				}
			break;

	default:   // Passes it on if unproccessed
	    return (DefWindowProc(hWnd, message, wParam, lParam));

	}

    return (0L);
	}



