// CompassWnd.c
// This file contains the window procedure and code related to the
// Compass window. This is also an OpenGL Window and displays a 
// directional Compass


#include <windows.h>	// Normal Windows stuff
#include <math.h>
#include <gl\gl.h>
#include <gl\glu.h>
#include "externs.h"	// Data shared between files
#include "glutils.h"



///////////////////////////////////////////////////////////////////////////////
// Setup the Rendering context for the Compass window
void SetupCompassRC(void)
	{
	// Black background
	glClearColor(0.0f, 0.0f, 0.0f, 1.0f);

	// Green drawing color
	glRGB(0,255,0);

	// Just draw the outlines of the polygons
	glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
	}



////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
// Window procedure, handles all messages for this window
LRESULT CALLBACK WndProcCompass(HWND    hWnd,
							UINT    message,
							WPARAM  wParam,
							LPARAM  lParam)
	{
	static HDC	  hDC;	// Keep the Device Context
	static HGLRC  hRC;	// Keep the Rendering Context

	switch (message)
		{
		// Window creation, setup here
		case WM_CREATE:
			// Save the device context
			hDC = GetDC(hWnd);

			// Set the pixel format
			SetDCPixelFormat(hDC);

			// Create palette if needed, this will return NULL if 
			// no palette is required
			hPalette = GetOpenGLPalette(hDC);
	
			// Create the rendering context and make it current
			hRC = wglCreateContext(hDC);
			wglMakeCurrent(hDC, hRC);

			// Do some setup here
			SetupCompassRC();

			wglMakeCurrent(hDC,NULL);
			break;

		// Window is being destroyed, cleanup
		case WM_DESTROY:
			// Cleanup...
			// Deselect the current rendering context and delete it
			wglMakeCurrent(hDC,NULL);
			wglDeleteContext(hRC);

			// Destroy the palette if it was created 
			if(hPalette != NULL)
				DeleteObject(hPalette);

			// Release the device context
			ReleaseDC(hWnd,hDC);

			break;

		// Window is resized. Setup the Viewing transformation
		case WM_SIZE:
			{
			int nWidth,nHeight;

			nWidth = LOWORD(lParam);  // width of client area 
			nHeight = HIWORD(lParam); // height of client area 

			// Make this windows rendering context current
			wglMakeCurrent(hDC, hRC);

			// Set the viewport to be the entire window
		    glViewport(0, 0, nWidth, nHeight);


			glMatrixMode(GL_PROJECTION);
			glLoadIdentity();

			// Establishes a 2 by 2 square with zero in the center
			gluOrtho2D(-1.0, 1.0, -1.0, 1.0); 

			glMatrixMode(GL_MODELVIEW);
			glLoadIdentity();

			// Deselect this windows device context
			wglMakeCurrent(hDC,NULL);
			}
			break;

		case WM_PAINT:
			{
			GLfloat fDegrees;
			GLint	nDegrees;

			wglMakeCurrent(hDC, hRC);
			
			// Clear the window with current clearing color
			glClear(GL_COLOR_BUFFER_BIT);

			glPushMatrix();

			// Draw the fixed position needle (CCW winding)
			glBegin(GL_QUADS);
				glVertex2f(0.0f,0.8f);
				glVertex2f(0.1f, 0.0f);
				glVertex2f(0.0f, -0.4f);
				glVertex2f(-0.1f, 0.0f);
			glEnd();

			// Convert radians to degrees
			fDegrees =  57.2957795f * (GLfloat)(pObject->radsFromEast);

			// Offset because stored value is angle from due east
			fDegrees = 90.0f - fDegrees;
			glRotatef(fDegrees,0.0f, 0.0f, 1.0f);

			// Draw the outline of an N
			glBegin(GL_LINE_STRIP);
				glVertex2f(-0.05f, 0.85f);
				glVertex2f(-0.05f, 0.99f);
				glVertex2f(0.05f, 0.85f);
				glVertex2f(0.05f, 0.99f);
			glEnd();

			// Draw the outline of an E
			glBegin(GL_LINE_STRIP);
				glVertex2f(0.97f, 0.08f);
				glVertex2f(0.86f, 0.08f);
				glVertex2f(0.86f, -0.08f);
				glVertex2f(0.98f, -0.08f);
			glEnd();

			glBegin(GL_LINES);
				glVertex2f(0.93f, 0.0f);
				glVertex2f(0.86f, 0.0f);
			glEnd();

			// Draw the outline of a W
			glBegin(GL_LINE_STRIP);
				glVertex2f(-0.99f, 0.05f);
				glVertex2f(-0.95f, -0.05f);
				glVertex2f(-0.90f, 0.05f);
				glVertex2f(-0.86f, -0.05f);
				glVertex2f(-0.81f, 0.05f);
			glEnd();

			// Draw the outline of an S
			glBegin(GL_LINE_STRIP);
				glVertex2f(0.05f, -0.84f);
				glVertex2f(-0.05f, -0.84f);
				glVertex2f(-0.05f, -0.90f);
				glVertex2f(0.05f, -0.90f);
				glVertex2f(0.05f, -0.96f);
				glVertex2f(-0.06f,-0.96f);
			glEnd();


			// Draw circular tick marks
			for(nDegrees = 0; nDegrees < 360; nDegrees += 10)
				{
				glPushMatrix();
					glRotatef((GLfloat)nDegrees,0.0f, 0.0f, 1.0f);

					glBegin(GL_LINES);
						glVertex2f(0.0f, 0.70f);

						// Longer tick marks at NSEW	
						if(nDegrees == 0 || nDegrees == 90 || nDegrees == 180 || nDegrees == 270)
							glVertex2f(0.0f, 0.82f);
						else
							glVertex2f(0.0f, 0.77f);
					
					glEnd();
					
				glPopMatrix();
				}
			
			glPopMatrix();
			glFlush();
						
			SwapBuffers(hDC);
			wglMakeCurrent(hDC,NULL);

			// Validate the newly painted client area
			ValidateRect(hWnd,NULL);
			}
			break;

		// Windows is telling the application that it may modify
		// the system palette.  This message in essance asks the 
		// application for a new palette.
		case WM_QUERYNEWPALETTE:
			// If the palette was created.
			if(hPalette)
				{
				int nRet;

				// Selects the palette into the current device context
				SelectPalette(hDC, hPalette, FALSE);

				// Map entries from the currently selected palette to
				// the system palette.  The return value is the number 
				// of palette entries modified.
				nRet = RealizePalette(hDC);

				// Repaint, forces remap of palette in current window
				InvalidateRect(hWnd,NULL,FALSE);

				return nRet;
				}
			break;

	
		// This window may set the palette, even though it is not the 
		// currently active window.
		case WM_PALETTECHANGED:
			// Don't do anything if the palette does not exist, or if
			// this is the window that changed the palette.
			if((hPalette != NULL) && ((HWND)wParam != hWnd))
				{
				// Select the palette into the device context
				SelectPalette(hDC,hPalette,FALSE);

				// Map entries to system palette
				RealizePalette(hDC);
				
				// Remap the current colors to the newly realized palette
				UpdateColors(hDC);
				return 0;
				}
			break;

	
		default:   // Passes it on if unproccessed
		    return (DefWindowProc(hWnd, message, wParam, lParam));
		}

    return (0L);
	}
